import { useState } from "react";
import DashboardLayout from "@/components/DashboardLayout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Phone, Search, Download, Play } from "lucide-react";

export default function CallHistory() {
  const [searchQuery, setSearchQuery] = useState("");
  const [filterStatus, setFilterStatus] = useState("all");

  // TODO: Remove mock data
  const calls = [
    { id: "1", date: "2025-01-10", time: "14:32", customer: "+1 (555) 123-4567", duration: "3:24", sentiment: "positive", status: "completed", scenario: "Banking" },
    { id: "2", date: "2025-01-10", time: "13:15", customer: "+1 (555) 234-5678", duration: "2:15", sentiment: "neutral", status: "completed", scenario: "Healthcare" },
    { id: "3", date: "2025-01-09", time: "16:45", customer: "+1 (555) 345-6789", duration: "4:52", sentiment: "positive", status: "completed", scenario: "Retail" },
    { id: "4", date: "2025-01-09", time: "11:20", customer: "+1 (555) 456-7890", duration: "1:48", sentiment: "negative", status: "escalated", scenario: "Insurance" },
    { id: "5", date: "2025-01-08", time: "09:30", customer: "+1 (555) 567-8901", duration: "5:12", sentiment: "positive", status: "completed", scenario: "Banking" }
  ];

  return (
    <DashboardLayout pageTitle="Call History">
      <div className="space-y-6">
        <Card>
          <CardHeader>
            <CardTitle>Search & Filter</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                <Input
                  placeholder="Search by phone number..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9"
                  data-testid="input-search"
                />
              </div>
              <Select value={filterStatus} onValueChange={setFilterStatus}>
                <SelectTrigger data-testid="select-status">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  <SelectItem value="completed">Completed</SelectItem>
                  <SelectItem value="escalated">Escalated</SelectItem>
                </SelectContent>
              </Select>
              <Button variant="outline" data-testid="button-export">
                <Download className="w-4 h-4 mr-2" />
                Export CSV
              </Button>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Call Records</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {calls.map((call) => (
                <div key={call.id} className="flex items-center justify-between p-4 rounded-md hover-elevate border border-border" data-testid={`call-${call.id}`}>
                  <div className="flex items-center gap-4 flex-1">
                    <div className="w-10 h-10 rounded-full bg-gradient-to-br from-indigo-600/20 to-cyan-600/20 flex items-center justify-center">
                      <Phone className="w-5 h-5 text-indigo-400" />
                    </div>
                    <div className="flex-1">
                      <div className="font-medium">{call.customer}</div>
                      <div className="text-sm text-muted-foreground">
                        {call.date} at {call.time} • {call.duration} • {call.scenario}
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-3">
                    <Badge variant={call.sentiment === "positive" ? "default" : call.sentiment === "negative" ? "destructive" : "secondary"}>
                      {call.sentiment}
                    </Badge>
                    <Badge variant={call.status === "completed" ? "default" : "destructive"}>
                      {call.status}
                    </Badge>
                    <Button size="sm" variant="ghost" data-testid={`button-play-${call.id}`}>
                      <Play className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
}
