import { useState } from "react";
import DashboardLayout from "@/components/DashboardLayout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { CreditCard, Download, Check, ArrowUpRight } from "lucide-react";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from "recharts";

export default function Billing() {
  const [currentPlan] = useState("Professional");

  // TODO: Remove mock data
  const usageData = [
    { month: "Jul", calls: 420, cost: 168 },
    { month: "Aug", calls: 510, cost: 204 },
    { month: "Sep", calls: 480, cost: 192 },
    { month: "Oct", calls: 550, cost: 220 },
    { month: "Nov", calls: 620, cost: 248 },
    { month: "Dec", calls: 580, cost: 232 }
  ];

  const invoices = [
    { id: "INV-001", date: "2025-01-01", amount: "$199.00", status: "paid" },
    { id: "INV-002", date: "2024-12-01", amount: "$199.00", status: "paid" },
    { id: "INV-003", date: "2024-11-01", amount: "$199.00", status: "paid" },
    { id: "INV-004", date: "2024-10-01", amount: "$199.00", status: "paid" }
  ];

  const plans = [
    {
      name: "Starter",
      price: "$49",
      features: ["500 calls/month", "Basic analytics", "Email support", "2 team members"],
      current: false
    },
    {
      name: "Professional",
      price: "$199",
      features: ["2,000 calls/month", "Advanced analytics", "Priority support", "10 team members", "Custom scenarios", "API access"],
      current: true
    },
    {
      name: "Enterprise",
      price: "$499",
      features: ["10,000 calls/month", "Custom analytics", "24/7 phone support", "Unlimited team members", "Custom integrations", "Dedicated account manager", "SLA guarantee"],
      current: false
    }
  ];

  return (
    <DashboardLayout pageTitle="Billing & Usage">
      <div className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Current Plan</CardTitle>
              <CreditCard className="w-4 h-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold" data-testid="current-plan">{currentPlan}</div>
              <p className="text-xs text-muted-foreground mt-1">$199/month</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">This Month</CardTitle>
              <ArrowUpRight className="w-4 h-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold" data-testid="current-usage">1,245</div>
              <p className="text-xs text-muted-foreground mt-1">calls used of 2,000</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Next Bill</CardTitle>
              <Download className="w-4 h-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold" data-testid="next-bill">$199.00</div>
              <p className="text-xs text-muted-foreground mt-1">Due Feb 1, 2025</p>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Usage Overview</CardTitle>
            <CardDescription>Your monthly usage and billing trend</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="mb-4">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium">Monthly Quota</span>
                <span className="text-sm text-muted-foreground">1,245 / 2,000 calls</span>
              </div>
              <Progress value={62} className="h-2" />
            </div>
            <ResponsiveContainer width="100%" height={250}>
              <LineChart data={usageData}>
                <CartesianGrid strokeDasharray="3 3" className="stroke-border" />
                <XAxis dataKey="month" className="text-xs" />
                <YAxis yAxisId="left" className="text-xs" />
                <YAxis yAxisId="right" orientation="right" className="text-xs" />
                <Tooltip contentStyle={{ backgroundColor: 'hsl(var(--card))', border: '1px solid hsl(var(--border))' }} />
                <Line yAxisId="left" type="monotone" dataKey="calls" stroke="hsl(var(--chart-1))" strokeWidth={2} name="Calls" />
                <Line yAxisId="right" type="monotone" dataKey="cost" stroke="hsl(var(--chart-2))" strokeWidth={2} name="Cost ($)" />
              </LineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Available Plans</CardTitle>
            <CardDescription>Choose the plan that fits your needs</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              {plans.map((plan, index) => (
                <Card key={plan.name} className={plan.current ? 'border-indigo-600/50' : ''}>
                  <CardHeader>
                    <div className="flex items-center justify-between">
                      <CardTitle className="text-lg">{plan.name}</CardTitle>
                      {plan.current && <Badge>Current</Badge>}
                    </div>
                    <div className="mt-2">
                      <span className="text-3xl font-bold">{plan.price}</span>
                      <span className="text-muted-foreground">/month</span>
                    </div>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <ul className="space-y-2">
                      {plan.features.map((feature, idx) => (
                        <li key={idx} className="flex items-center gap-2 text-sm">
                          <Check className="w-4 h-4 text-green-500 shrink-0" />
                          <span>{feature}</span>
                        </li>
                      ))}
                    </ul>
                    <Button
                      className={`w-full ${plan.current ? '' : 'bg-gradient-to-r from-indigo-600 to-cyan-600 hover:from-indigo-500 hover:to-cyan-500'}`}
                      variant={plan.current ? "outline" : "default"}
                      disabled={plan.current}
                      data-testid={`button-plan-${index}`}
                    >
                      {plan.current ? "Current Plan" : "Upgrade"}
                    </Button>
                  </CardContent>
                </Card>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Billing History</CardTitle>
            <CardDescription>Your past invoices and payments</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {invoices.map((invoice) => (
                <div key={invoice.id} className="flex items-center justify-between p-3 rounded-md hover-elevate border border-border" data-testid={`invoice-${invoice.id}`}>
                  <div className="flex items-center gap-3">
                    <div className="w-10 h-10 rounded-md bg-gradient-to-br from-indigo-600/20 to-cyan-600/20 flex items-center justify-center">
                      <CreditCard className="w-5 h-5 text-indigo-400" />
                    </div>
                    <div>
                      <div className="font-medium">{invoice.id}</div>
                      <div className="text-sm text-muted-foreground">{invoice.date}</div>
                    </div>
                  </div>
                  <div className="flex items-center gap-3">
                    <div className="text-right">
                      <div className="font-medium">{invoice.amount}</div>
                      <Badge variant="default" className="text-xs">
                        {invoice.status}
                      </Badge>
                    </div>
                    <Button size="sm" variant="outline" data-testid={`button-download-${invoice.id}`}>
                      <Download className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
}
